unit HmRs232;
{ A Delphi (Pascal) mock program module containing what is necessary or
  helpful for controlling Homer via an RS232 interface.

  19-Dec-03  First version
  12-Mar-04  ReceiveDataObject
  18-Sep-06  DecodeMDO
  10-Aug-09  Added PrflIncluded, modified DecodeMDO
  29-Jan-10  Added DecodeMotorPositions; DecodeMDO modification
  12-Mar-19  Added explananion of function Dexp(x)
  18-Nov-22  Type Tfloat replaced by double. Slight overhaul.
}

interface

const
  { Constants }
  RXtimeout = 6; { The communication error byte (Err00) is assigned this value
    in the ReceiveByte function if an RS232 timeout has occurred }

  CmndLbl = 128;
  { Command Label: precedes a byte that should be interpreted as
    a command/message code instead of data (number). If the data byte 128 needs to be
    sent, then the byte is sent twice (128, 128). Examples:
    1. If you receive ... 128, 28, ... then 28 is the command byte msDataBegin
       saying that a stream of data bytes will follow until the next command byte
       occurs. In this mock unit, we are only interested in the data streams ending
       with 128, 16, which means that it is a Measurement Data Object (MDO).
    2. If you receive ... 128, 128, ... this represents a single data byte
       with Value = 128 }

  { Some command codes }
  msMeasObject = 16;
  msDataBegin = 28;

  { Other constants }
  MaxNbuf = 16384; { Max size of RX buffer. 16384 is a very conservative upper bound. }
  NdoMax = 31; { Max size of MDO }

type
  TBuf = array [0 .. MaxNbuf] of byte; { RX buffer structure }
  TMdo = array [0 .. NdoMax - 1] of byte;  { structure for MDO:
    array of NdoMax bytes }

  complex = record
    re, im: double
  end;

  THresults = record  { Structure containing convenient decoded data }
    HomerIncluded, MotorsIncluded: boolean;
    HomerValid: boolean; { true if Homer measurement results are valid }
    PulseSerialIncluded: boolean; { true if MDO is a sample of a Rectified
      or Pulsed waveform }
    PrflIncluded: boolean; { true if MDO includes mean reflected  power }
    Fre: longint; { Frequency in units of 10 Hz }
    Rho_m: complex; { Real+imag part of measured reflection coefficient }
    Mag_m, Pha_m: double; { Mag+phase of measured reflection coefficient }
    Rho_d: complex; { Load (deembedded) reflection coefficient }
    Mag_d, Pha_d: double;
    Pinc, Prfl, Pabs: double; { Incident, reflected and absorbed powers }
    Temp: double; { Temperature in Celsius }
    PulsErr: boolean; { Error: true if Pulse measurement has failed }
    Oflow: boolean; { Warning: true if Overflow of ADCs has occured }
    TooCold, TooHot: boolean; { Warning: true if Temperature limits were exceeded }
    LowSignal: boolean; { Warning: true if the signal level is too low }
    FreSubst: boolean; { Warning: true if the frequency measurement has failed
      and the substitute frequency has been used }
    FrePast: boolean; { Info: true if MDO contains a previously measured frequency }
    HmInvalid: boolean; { Error: true if Homer Analyzer data in MDO are invalid }
    PulseSerial: integer;  { Serial number of a Rectified or Pulsed sample }
  end;

var
  { Variables }
  Mdo: TMdo; { Measurement Data Object - MDO }
  Err00: integer; { Communication error }
  ErrByte: byte; { Sent by Homer as Mdo[1] if Homer Anlyzer data included in MDO }
  Mdo_size: word; { Byte count of the currently received MDO }
  HomResults: THresults; { Results obtained by decoding MDO }

implementation

var
  Finito: boolean; { flag that, when true, is used to stop receiving bytes }

function ReceiveByte: byte;
{ An example of the Windows receive byte routine. ReadFile and ClearCommError
  functions are Delphi implementations of Win API functions.
  This function waits for a byte from the serial link. If nothing arrives before
  a set Timeout, it sets Err00 error byte and returns your predefined
  substitute byte. Timeout is set by implementation of the SetCommTimeouts
  Win API function. Other Win API functions must be used beforehand
  to prepare COM port. These include CreateFile (which returns the file
  handle Hnd), GetCommState, SetCommState, etc. }
var
  ok: boolean;
  n: longint;
  rx_byte: byte;
begin
  Err00 := 0;
  { Wait for 1 byte; n means how many bytes have actually been read }
  ok := ReadFile(Hnd, rx_byte, 1, n, nil) and (n > 0);
  if ok then
    Result := rx_byte
  else
  begin { an error occured }
    Err00 := RXtimeout;
    ok := ClearCommError(Hnd, ComErr, ComStat); { Win API }
    Result := SubstituteByte;
  end;
end;

procedure ReceiveAndDecodeByte(var b: byte; var IsCommand: boolean);
{ This procedure receives a byte (B1). If B1 differs from the Command Label
  CmndLbl=128 then B1 is returned as b, IsCommand flag is set to false,
  then the routine ends. If B1 is equal to Command Label (B1=CmndLbl=128)
  then another byte (B2) will be received. If also B2=CmndLbl then
  b=CmndLbl=128 is returned as data byte and IsCommand flag is
  set to false. If B2 differs from B1, b=B2 is returned and should be
  interpreted as command code rather than a data byte. To indicate this,
  IsCommand flag is set to true }
begin
  IsCommand := false;
  b := ReceiveByte; { read one byte from RS232. ReceiveByte routine
    also sets error byte Err00 }
  if Err00 = 0 then
    if b = CmndLbl then
    begin { read again }
      b := ReceiveByte;
      IsCommand := b <> CmndLbl;
    end
end;

function ReceiveDataObject(wait_ms: integer; var RxBuff; var n: integer;
  var EndMsg: byte): boolean;
{ Wait at most wait_ms milliseconds for the arrival of msDataBegin message.
  If it arrives, the routine resets the timeout and keeps receiving data until
  the next command comes, which is interpreted as EndMsg }
var
  Buffer: TBuf absolute RxBuff; { a way of converting structure types }
  WasCommand, DataBeginDetected, Finish: boolean;
  b: byte;
  StartTime, Lasts: longint;
begin
  n := 0; { initialize data buffer counter }
  EndMsg := 255; { in case of failure, return a nonexistent EndMsg }
  DataBeginDetected := false; { set to true when msDataBegin command comes }
  Finish := false; { set this flag to true to quit this routine}
  StartTime := GetTickCount; { initialize timeout timer }
  repeat
    Result := false;
    ReceiveAndDecodeByte(b, WasCommand);
    if Err00 = 0 then
    begin { no comm error occured }
      if WasCommand then
      begin { byte b is a command not data }
        case b of
          msDataBegin: { the command is data begin: hence subsequent received
              bytes will be data. Receive this data until the next byte comes that is
              a command byte instead of a data byte, which is the end command code
              EndMsg }
            begin
              n := 0; { initialize data buffer counter }
              DataBeginDetected := true; { flag that, when true, indicates
                to store incoming data }
              StartTime := GetTickCount; { reset timeout }
            end;
        else
          if DataBeginDetected then
          begin { first command byte after msDataBegin is end command EndMsg;
            do finish with success }
            Finish := true;
            EndMsg := b;
            Result := true;
          end;
        end
      end { of WasCommand }
      else { byte b is data }
        if DataBeginDetected then
        begin
          { Store data provided msDataBegin came before }
          Buffer[n] := b;
          inc(n)
        end;
    end { of case Err00=0 }
    else
      Finish := true; { communication error }
    Application.ProcessMessages; { see MainLoopExample }
    Lasts := GetTickCount - StartTime; { how long it takes }
  until Finish or (Lasts > wait_ms); { end if success or error or timeout }
end; { ReceiveDataObject }

function ReceiveMeasObject: boolean;
{ This function receives bytes until the command byte msDataBegin arrives.
  Then it continues to receive data bytes and stores them to MDO array until
  an end command byte arrives. If the end command byte is msMeasObject, then
  data were indeed MDO and all is ok; return Result=true. If a communication
  error occured or the received data block was not MDO, then return Result=false }
var
  b: byte;
  WasCommand, Finish, DataBeginDetected: boolean;
  n: word;
begin
  b := 0;
  n := 0; { initialize counter of received bytes }
  Finish := false; { reset flag to quit }
  DataBeginDetected := false;
  { Repeat receiving and interpreting bytes }
  repeat
    { ReceiveAndDecodeByte routine receives a byte and decides whether it was
      a command byte or data byte. Its internal routines also set communication
      error Err00, which can be timeout error. In principle, a timeout decision
      could also be implemented in this cycle }
    ReceiveAndDecodeByte(b, WasCommand);
    if Err00 = 0 then
    begin { no comm error occured }
      if WasCommand then
      begin { byte b is a command not data }
        case b of { what command is it? }
          msDataBegin: { The command is data begin: hence the following received
              bytes will be data - but they may or may not be MDO data. Receive
              them until the next byte comes that is a command instead of data.
              This end command byte will say whether the data were MDO. Anyway,
              store first NdoMax of the received bytes to MDO }
            begin
              n := 0; { initialize counter of received bytes }
              DataBeginDetected := true { flag that, when true, indicates
                to store incoming data }
            end;
          msMeasObject:
            begin { This is a terminating command byte for a MDO. But we have
                received a complete data block only if msDataBegin command has
                arrived before, which is indicated by DataBeginDetected=true.
                In this case, we can finish with success. The other case means
                that we have missed a data begin (we started receiving bytes
                in the middle of transaction); we will exit (set Finish to true)
                but with no success.
                Note 1: An alternative way to deal with the latter (=no success)
                case is leaving Finish to remain false; then you just continue
                receiving bytes until a new msDataBegin command arrives. We have
                not tested this option. }
              Result := DataBeginDetected;
              Finish := true; { or, in accordance with Note 1, you may wish
                to modify this command to Finish := DataBeginDetected }
            end;
        end { case }
      end
      else { b is data byte. If msDataBegin command came before, store it }
        if DataBeginDetected then
        begin
          if n < NdoMax then
          begin { store up to NdoMax data bytes }
            Mdo[n] := b; { store }
            inc(n) { increase counter }
          end
          else
          begin { if more than NdoMax data came, this is definitely not an MDO,
              finish with no success }
            Result := false;
            Finish := true; { or you can leave it false to continue receiving
              and try to catch the next msDataBegin command byte }
          end;
        end
    end
    else
    begin { comm error occured }
      Finish := true; { quit }
      Result := false; { with the result no success }
    end;
  until Finish;
  { Fill Status Byte SB (this command can be located elsewhere as well) }
  if Result then
    StatusByte := Mdo[0];
end; { HM_ReceiveMeasObjec }

function CheckSumCorrect(var Mo: TMdo; n: integer): boolean;
{ Verify the checksum of object Mo with size n; return true if ok }
var
  i, Csum: integer;
begin
  Csum := 0;
  { Valid Mo bytes are Mo[0]...Mo[n-1], the last one contains the sent checksum
    byte: we must not use it for our checksum computations }
  for i := 0 to n - 2 do
    Csum := Csum + Mo[i];
  Csum := Csum and 255; { Take only LSB }
  Result := Csum = Mo[n - 1]; { Compare }
end;

procedure AnalyzeErrorByte;
{ Evaluate errors. For details, please refer to THresults type definition }
begin
  with HomResults do
  begin
    PulsErr := ErrByte and 1 = 1;     { Pulse measurement has failed }
    Oflow := ErrByte and 2 = 2;       { Overflow of ADCs }
    TooCold := ErrByte and 4 = 4;     { Temperature limits exceeded }
    TooHot := ErrByte and 8 = 8;
    LowSignal := ErrByte and 16 = 16; { Warning: low signal }
    FreSubst := ErrByte and 32 = 32;  { Warning: substitute frequency used }
    HmInvalid := ErrByte and 64 = 64; { Analyzer data for some reason invalid }
  end;
end;

function DecodeMDO: boolean;
var
  i: integer;
  Mdo_index: word;

  { Typecasting procedures in Delphi notation }
  function GetSmallPtr(p: pointer): smallint; { signed 16-bit }
  type
    PSmall = ^smallint;
  begin
    { Convert 2 bytes starting at pointer p to a signed 16-bit integer }
    GetSmallPtr := PSmall(p)^
  end;

  function GetLongPtr(p: pointer): longint; { signed 32-bit }
  type
    PLong = ^longint;
  begin
    { Convert 4 bytes starting at pointer p to a signed 32-bit integer }
    GetLongPtr := PLong(p)^
  end;

  function GetWordPtr(p: pointer): word; { unsigned 16-bit }
  type
    PWord = ^word;
  begin
    { Convert 2 bytes starting at pointer p to an unsigned 16-bit integer }
    GetWordPtr := PWord(p)^
  end;

  procedure DecodeMeasurementData;
  begin
    with HomResults do
    begin
      { Meaning see in THresults type definition }
      Pinc := (256 * word(Mdo[2]) + Mdo[3]) * Dexp(Mdo[4] - 10);
      { Function Dexp(x) = 10 to power x, e.g. Dexp(3) = 1000 }
      Temp := GetSmallPtr(@Mdo[5]) / 10.0;
      Rho_m.re := GetSmallPtr(@Mdo[8]) / 4096.0;
      Rho_m.im := GetSmallPtr(@Mdo[10]) / 4096.0;
      Fre := GetLongPtr(@Mdo[12]);
      Rho_d.re := GetSmallPtr(@Mdo[16]) / 4096.0;
      Rho_d.im := GetSmallPtr(@Mdo[18]) / 4096.0;
      inc(Mdo_index, 19);
      if PulseSerialIncluded then
      begin
        PulseSerial := GetWordPtr(@Mdo[Mdo_index]);
        inc(Mdo_index, 2); { Mdo_index will be 22 }
      end
      else if PrflIncluded then
      begin
        Prfl := (256 * word(Mdo[21]) + Mdo[20]) * Dexp(Mdo[7] - 10);
        inc(Mdo_index, 2); { Mdo_index will be 22 }
      end;
    end;
  end;

  procedure DecodeMotorPositions;
  begin
    Mot1_steps := GetSmallPtr(@Mdo[Mdo_index]);
    Mot2_steps := GetSmallPtr(@Mdo[Mdo_index + 2]);
    Mot3_steps := GetSmallPtr(@Mdo[Mdo_index + 4]);
    MotStatByte1 := Mdo[Mdo_index + 6];
    MotStatByte2 := Mdo[Mdo_index + 7];
  end;

  procedure DerivedResults;
  begin
    with HomResults do
    begin
      { For details, please refer to THresults type definition }
      Mag_m := Hypot(Rho_m.re, Rho_m.im);  { Magnitude }
      Pha_m := ArcTan2(Rho_m.im, Rho_m.re) * 180.0 / Pi; { Phase:
        Never use Atan(im/re)!!! }
      Mag_d := Hypot(Rho_d.re, Rho_d.im);
      Pha_d := ArcTan2(Rho_d.im, Rho_d.re) * 180.0 / Pi;
      if not PrflIncluded then
        Prfl := Pinc * sqr(Mag_m);
      if Prfl > Pinc then
        Prfl := Pinc; { Clipping - passsive loads assumed }
      Pabs := Pinc - Prfl;
    end;
  end;

begin { DecodeMDO }
  { First: determine number of bytes Mdo_size for the ability to verify
    the checksum, which is stored in the last byte }
  with HomResults do
  begin
    { At least one byte is always transmitted = status Mdo[0] }
    Mdo_size := 1;
    { Homer measurement data are present if bit 2 of Mdo[0] is 1 }
    HomerIncluded := Mdo[0] shr 2 and 1 = 1;
    if HomerIncluded then { another 19 bytes are present }
    begin
      inc(Mdo_size, 19);
      ErrByte := Mdo[1];
      AnalyzeErrorByte(HomResults);
    end;
    PulseSerialIncluded := HomerIncluded and ((Mdo[0] and 3) <> 0);
    PrflIncluded := (not PulseSerialIncluded) and
      (HomerIncluded and ((Mdo[0] and 64) > 0));
    if PrflIncluded or PulseSerialIncluded then
      inc(Mdo_size, 2);
    { Motors data are present if bit 4 of Mdo[0] is 1 }
    MotorsIncluded := Mdo[0] shr 4 and 1 = 1;
    if MotorsIncluded then
      inc(Mdo_size, 8); { another 8 bytes are present }
    HomerValid := HomerIncluded and not HmInvalid;
    inc(Mdo_size);
    { last byte is checksum }
    Result := CheckSumCorrect(Mdo, Mdo_size) and (HomerValid or MotorsIncluded);
    { Previous frequency is sent, i.e. no newly measured value exists }
    FrePast := Mdo[0] shr 3 and 1 = 1;
    if not Result then
      { exit to avoid various exceptions like division by zero }
      exit;
      { Now that various flags have been determined and data have been checked
        for integrity (checksum) and correctness (not HmInvalid),
        perform the decoding  }
    Mdo_index := 1; { temporary pointer to MDO bytes }
    if HomerIncluded then
    begin
      if HomerValid then
      begin
        DecodeMeasurementData;
        DerivedResults;
      end
      else
      begin { do not decode; leave previous results }
        if PulseSerialIncluded or PrflIncluded then
          inc(Mdo_index, 21)
        else
          inc(Mdo_index, 19)
      end;
    end;
    if MotorsIncluded then
      DecodeMotorPositions;  { uses the current value of Mdo_index }
  end; { with HomResults }
end;

procedure MainLoopExample;
{ Just an example of how it could be }
var
  ok, HomerIncluded, MotorsIncluded: boolean;
begin
  { Clear Windows RS232  buffers - Delphi implementation of Windows API function }
  PurgeComm(Hnd, PURGE_TXABORT + PURGE_TXCLEAR + PURGE_RXABORT + PURGE_RXCLEAR);
  Finito := false; { clear the flag }
  repeat { main loop }
    ok := ReceiveMeasObject; { receive MDO }
    { Function DecodeMDO takes an MDO as implicit input and converts its bytes to
      desired quantities. The obtained results are returned in the structure
      HomResults. In this example, the DecodeMDO function also returns flags
      HomerIncluded and MotorsIncluded saying whether valid Homer Analyzer data
      and motors data were present in MDO, respectively }
    if ok then
      ok := DecodeMDO;
    { Now process the obtained data as you wish }
    if ok then
    begin
      if HomerIncluded then
        DoWhatYouWishWithHomerData;
      if MotorsIncluded then
        DoWhatYouWishWithMotorsData;
    end;
    { Application.ProcessMessages is a Delphi function that provides Windows
      a chance to respond to incoming messages that are waiting to be attended to.
      It may be, e.g., a message saying "User pressed Escape key
      and wishes the whole program to be stopped". To stop the whole program,
      User has to write a routine termed event handler which performs
      some action if Escape (or other desired key) has been pressed. In a Delphi
      environment, the event of pressing a key is called a KeyDown event
      and the handler's name which is called automatically if this event
      occurs is FormKeyDown (see below). In our example, the handler only
      sets the flag Finito to true }
    Application.ProcessMessages;
  until Finito;
end;

procedure FormKeyDown(Sender: TObject; var Key: word; Shift: TShiftState);
begin
  case Key of { which key has been pressed }
    VK_ESCAPE:
      Finito := true;
    VK_RIGHT:
      PutYourHandToYourHead;
    VK_SPACE:
      DoWhateverSimonSays;
    { etc... }
    end;

end { of the module }.
